﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Microsoft.Win32;

using MicroRWD.Common;
using MicroRWD.QT;

namespace MicroRWD.UI.QT
{
    public partial class Controller : Form
    {
        #region Private Constants

        // Default command timeout in milliseconds
        private const int C_DEFAULT_COMMAND_TIMEOUT_MS = 1000;

        // Default mode
        private const Mode C_DEFAULT_MODE = Mode.HITAG1_S;

        // Base registry key (for configuration)
        private const string C_USER_SUBKEY = @"Software\IB Technology\MicroRWD";

        #endregion


        #region Public Properties

        // The configured command timeout (stored in the registry)
        public int CommandTimeoutMillis
        {
            get
            {
                return (int)Registry.CurrentUser.CreateSubKey(C_USER_SUBKEY).GetValue("CommandTimeoutMillis", C_DEFAULT_COMMAND_TIMEOUT_MS);
            }

            private set
            {
                Registry.CurrentUser.CreateSubKey(C_USER_SUBKEY).SetValue("CommandTimeoutMillis", value);
            }
        }

        // The configured port name (stored in the registry)
        public string PortName
        {
            get
            {
                return (string) Registry.CurrentUser.CreateSubKey(C_USER_SUBKEY).GetValue("PortName", "COM1");
            }

            private set
            {
                if (!String.IsNullOrEmpty(value))
                {
                    Registry.CurrentUser.CreateSubKey(C_USER_SUBKEY).SetValue("PortName", value);
                }
            }
        }

        // Quad tag reader
        public QuadTagReader Reader { get; private set; }

        #endregion


        #region Constructor

        // Construct a controller instance
        public Controller()
        {
            // Initialise UI components
            InitializeComponent();

            // Initialise command timeout on toolstrip menu
            UpdateCommandTimeoutToolStripMenuItem();

            // Initialise reader
            Reader = new QuadTagReader(PortName, CommandTimeoutMillis);

            // Start the reader thread
            Reader.Start();

            // Select card type panel
            SelectCardTypePanel();
        }

        #endregion


        #region Public Methods

        // Select the card type panel
        public void SelectCardTypePanel()
        {
            // Disable 'view' menu item
            this.UIThread(() => viewToolStripMenuItem.Enabled = true);
            this.UIThread(() => cardTypeToolStripMenuItem.Checked = true);
            this.UIThread(() => dataViewToolStripMenuItem.Checked = false);

            // Ensure reader is in continuous mode
            Reader.ContinuousPoll = true;

            // Make card type panel visible, data view panel invisible
            this.UIThread(() => cardPanel.SetVisible(true));
            this.UIThread(() => dataPanel.SetVisible(false));
        }

        // Select the data view panel
        public void SelectDataViewPanel(Mode _mode)
        {
            // Enable 'view' menu item
            this.UIThread(() => viewToolStripMenuItem.Enabled = true);
            this.UIThread(() => dataViewToolStripMenuItem.Checked = true);
            this.UIThread(() => cardTypeToolStripMenuItem.Checked = false);

            // Ensure reader is in continuous mode
            Reader.ContinuousPoll = true;

            // Make data view panel visible, card type panel invisible
            this.UIThread(() => dataPanel.SetVisible(true));
            this.UIThread(() => cardPanel.SetVisible(false));
        }

        // Exit application
        public void Exit()
        {
            // Terminate all event loops and return control to Program.cs
            Application.Exit();
        }

        #endregion


        #region Private Functions

        // Update the command timeout toolstrip menu items to reflect current value
        private void UpdateCommandTimeoutToolStripMenuItem()
        {
            // Update command timeout toolstrip menu items checked state
            this.UIThread(() => ct20msToolStripMenuItem.Checked = int.Parse((string)ct20msToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct40msToolStripMenuItem.Checked = int.Parse((string)ct40msToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct80msToolStripMenuItem.Checked = int.Parse((string)ct80msToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct160msToolStripMenuItem.Checked = int.Parse((string)ct160msToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct320msToolStripMenuItem.Checked = int.Parse((string)ct320msToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct540msToolStripMenuItem.Checked = int.Parse((string)ct540msToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct1sToolStripMenuItem.Checked = int.Parse((string)ct1sToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct2sToolStripMenuItem.Checked = int.Parse((string)ct2sToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct4sToolStripMenuItem.Checked = int.Parse((string)ct4sToolStripMenuItem.Tag) == CommandTimeoutMillis);
            this.UIThread(() => ct8sToolStripMenuItem.Checked = int.Parse((string)ct8sToolStripMenuItem.Tag) == CommandTimeoutMillis);
        }

        #region Event Handling

        // Handle exit menu item click
        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Exit application
            Exit();
        }

        // Handle com port menu item click
        private void comPortToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Allow user to configure COM port passing in current property setting
            ComPortForm comPortForm = new ComPortForm(PortName);
            DialogResult dr = comPortForm.ShowDialog();

            // If user pressed OK then update Port property setting
            if (dr == DialogResult.OK)
            {
                // Only act on changes
                if (!String.Equals(PortName, comPortForm.PortName))
                {
                    // Update property
                    PortName = comPortForm.PortName;

                    // Update reader
                    Reader.PortName = PortName;
                }
            }
        }

        // Invoked when controller is closing, used to shutdown the reader instance
        private void MicroRWD_Controller_FormClosing(object sender, FormClosingEventArgs e)
        {
            // Shutdown the reader thread
            Reader.Stop();

            // Release resources allocated to reader
            Reader.Dispose();
        }

        // Handle command timeout menu items click
        private void ctToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Get sender
            ToolStripMenuItem item = (ToolStripMenuItem)sender;

            // Update Command Timeout property setting
            CommandTimeoutMillis = int.Parse((string)item.Tag);

            // Update current setting in reader
            Reader.CommandTimeoutMillis = CommandTimeoutMillis;

            // Update UI
            UpdateCommandTimeoutToolStripMenuItem();
        }

        // Handle help about menu item click
        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Instantiate help about dialog
            HelpAboutForm form = new HelpAboutForm();
            form.ID_Text = Reader.CmdMessage();
            form.ShowDialog();
        }

        // Handle help website menu item click
        private void websiteToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Launch default browser to open IB Technology website
            try
            {
                System.Diagnostics.Process.Start("http://www.ibtechnology.co.uk/");
            }
            catch (System.ComponentModel.Win32Exception ex)
            {
                    MessageBox.Show(this, ex.Message, "Exception", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            catch (System.Exception ex)
            {
                MessageBox.Show(this, ex.Message, "Exception", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        // Handle help manual menu item click
        private void manualToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Launch default PDF reader to open associated datasheet / manual
            try
            {
                System.Diagnostics.Process.Start("RWD_QT.PDF");
            }
            catch (System.ComponentModel.Win32Exception ex)
            {
                MessageBox.Show(this, ex.Message, "Exception", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            catch (System.Exception ex)
            {
                MessageBox.Show(this, ex.Message, "Exception", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        // Handle card type panel menu item click
        private void cardTypeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Select card type panel
            SelectCardTypePanel();
        }

        // Handle data view panel menu item click
        private void dataViewToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // Select date view panel
            if (Reader.Mode != Mode.NONE)
            {
                SelectDataViewPanel(Reader.Mode);
            }
        }

        #endregion

        #endregion
    }
}
